import {DateHelper} from "../helper/DateHelper";
import {MonthData} from "./MonthData"
import {DayData} from "./DayData"

export class CalendarData {
	monthCount: number
	startYear: number
	startMonth: number
	monthDataList: Array<MonthData>

	/**
	 * @param monthCount 月份个数
	 * @param startYear 起始年
	 * @param startMonth 起始月
	 */
	constructor(monthCount: number, startYear?: number, startMonth?: number) {
		this.monthCount = monthCount;
		this.startYear = startYear && startYear >= 1970 ? startYear : new Date().getFullYear();
		this.startMonth = startMonth && startMonth > 0 && startMonth <= 12 ? startMonth : new Date().getMonth() + 1;

		const months = new Array<MonthData>();
		let monthData = CalendarData.getCalendarMonth(this.startYear, this.startMonth);
		months.push(monthData)
		for (let i = 1; i < monthCount; i++) {
			monthData = CalendarData.getNextMonth(monthData.year, monthData.month);
			months.push(monthData)
		}
		this.monthDataList = months;
	}

	/**
	 * 获取当前时间 （2018-08-15）
	 */
	static getCurrentFormatDateString(): string {
		const date = new Date();
		return date.getFullYear() + "-" + DateHelper.formatNumber(date.getMonth() + 1) + "-" + DateHelper.formatNumber(date.getDate());
	}

	/**
	 * 获取日历数据
	 * @param year
	 * @param month
	 */
	static getCalendarMonth(year: number, month: number): MonthData {
		const monthInfo = new MonthData(year, month);
		const weekDay = DateHelper.getWeekDay(year, month);
		if (weekDay > 0) {
			//月份的第一天不是表的第一天。需要在前面插入上个月数据
			const lastMonthInfo = DateHelper.getLastMonthInfo(year, month);
			const lastMonthDays = DateHelper.getMonthDays(lastMonthInfo.year, lastMonthInfo.month)
			for (let j = 0; j < weekDay; j++) {
				const currentDay = lastMonthDays - weekDay + 1 + j;
				const dayInfo = new DayData(lastMonthInfo.year, lastMonthInfo.month, currentDay);
				dayInfo.isSupplementDay = true;
				monthInfo.dayItems.unshift(dayInfo)
			}
		}
		const size = monthInfo.dayItems.length;
		//为了列表显示（个数必须是7的倍数），必须补充空数据
		const emptyDays = 7 - size % 7;
		if (emptyDays < 7) {
			const nextMonthInfo = DateHelper.getNextMonthInfo(year, month)
			for (let j = 0; j < emptyDays; j++) {
				let dayInfo = new DayData(nextMonthInfo.year, nextMonthInfo.month, (j + 1));
				dayInfo.isSupplementDay = true;
				monthInfo.dayItems.push(dayInfo)
			}
		}
		return monthInfo;
	}

	/**
	 * 获取上个月的数据
	 * @param year
	 * @param month
	 */
	static getLastMonthData(year: number, month: number): MonthData {
		const info = DateHelper.getLastMonthInfo(year, month);
		return CalendarData.getCalendarMonth(info.year, info.month)
	}

	/**
	 * 获取下个月的数据
	 * @param year
	 * @param month
	 */
	static getNextMonth(year: number, month: number): MonthData {
		const info = DateHelper.getNextMonthInfo(year, month);
		return CalendarData.getCalendarMonth(info.year, info.month)
	}

	/**
	 * 获取当前月份的日历数据
	 */
	static getCurrentMonth(): MonthData {
		const date = new Date();
		return CalendarData.getCalendarMonth(date.getFullYear(), date.getMonth() + 1)
	}

	/**
	 * 获取日历基础数据(从当前月，往后推)
	 * @param count
	 * @returns {Array}
	 */
	static getCalendarMonthList(count: number): Array<MonthData> {
		const months = [];
		let data = CalendarData.getCurrentMonth();
		months.push(data)
		for (let i = 1; i < count; i++) {
			data = CalendarData.getNextMonth(data.year, data.month);
			months.push(data)
		}
		return months;
	}
}
