import {dateConstants} from './DateConstants'

class CountDownInfo {
  time: number;
  day: number;
  hour: number;
  minute: number;
  second: number;

  constructor(time: number) {
    let hourTime = time % dateConstants.dayMillis;//剩余的小时部分时间
    let minuteTime = time % dateConstants.hourMillis;//剩余的分钟部分时间
    let secondTime = time % dateConstants.minuteMillis;//剩余的秒部分时间
    this.time = time;
    this.day = Math.floor(time / dateConstants.dayMillis);
    this.hour = Math.floor(hourTime / dateConstants.hourMillis);
    this.minute = Math.floor(minuteTime / dateConstants.minuteMillis);
    this.second = Math.floor(secondTime / dateConstants.secondMillis);
  }

  format(value: number): string {
    return value >= 10 ? value + '' : "0" + value;
  }
}

type CountDownFinishCallback = () => void;

type CountDownCallback = (info: CountDownInfo) => void;

export class CountDownTimer {
  private millisInFuture: number;
  private readonly mInterval: number;
  private onCountDownFinishListener: CountDownFinishCallback;
  private onCountDownListener: CountDownCallback;
  private timeOutId: NodeJS.Timeout;

  /**
   * @param millisInFuture 时间毫秒数
   * @param countDownInterval 时间毫秒数（单位间隔）
   */
  constructor(millisInFuture: number, countDownInterval: number) {
    this.millisInFuture = millisInFuture;
    this.mInterval = countDownInterval;
    this.onCountDownFinishListener = undefined;
    this.onCountDownListener = undefined;
    this.timeOutId = undefined;
  }

  addCountDownListener(listener: CountDownCallback): void {
    if (listener && typeof listener === 'function') {
      this.onCountDownListener = listener;
    }
  }

  addFinishListener(listener: CountDownFinishCallback): void {
    if (listener && typeof listener === 'function') {
      this.onCountDownFinishListener = listener;
    }
  }

  start(): void {
    if (!this.timeOutId && this.onCountDownListener != undefined) {
      this.timeOutId = setInterval(() => {
        this.millisInFuture = this.millisInFuture - this.mInterval;
        if (this.millisInFuture > 0) {
          this.onCountDownListener(CountDownTimer.getTimeInfo(this.millisInFuture));
        } else {
          this.cancel();
          if (this.onCountDownFinishListener !== undefined) {
            this.onCountDownFinishListener();
          }
        }
      }, this.mInterval);
    }
  }

  cancel(): void {
    if (this.timeOutId) {
      clearTimeout(this.timeOutId);
      this.timeOutId = undefined;
    }
  }

  static getTimeInfo(time: number): CountDownInfo {
    return new CountDownInfo(time);
  }
}


