import {dateConstants, DateType, MonthDayNumber} from "./DateConstants"
///yyyy-MM-dd HH:mm:ss
const formatDateList = [
	{
		symbolList: ["yyyy", "YYYY"],
		getValue: function (date: Date): string {
			return DateHelper.formatNumber(date.getFullYear());
		}
	},
	{
		symbolList: ["MM"],
		getValue: function (date: Date): string {
			return DateHelper.formatNumber(date.getMonth() + 1);
		}
	},
	{
		symbolList: ["DD", "dd"],
		getValue: function (date: Date): string {
			return DateHelper.formatNumber(date.getDate());
		}
	},
	{
		symbolList: ["HH", "hh"],
		getValue: function (date: Date): string {
			return DateHelper.formatNumber(date.getHours());
		}
	},
	{
		symbolList: ["mm"],
		getValue: function (date: Date): string {
			return DateHelper.formatNumber(date.getMinutes());
		}
	},
	{
		symbolList: ["SS", "ss"],
		getValue: function (date: Date): string {
			return DateHelper.formatNumber(date.getSeconds());
		}
	},
]

export class DateHelper {
	/**
	 * 获取日期实例
	 * 兼容ios new Date('2020-10-10') 返回Invalid Date
	 * @param date
	 */
	static getDateInstance(date: DateType): Date {
		if (date instanceof Date) {
			return date;
		}
		if (typeof date === 'string' && date.indexOf('-') > 0) {//兼容ios new Date('2020-10-10') 返回Invalid Date
			const reg = new RegExp("-", "g");
			date = date.replace(reg, '/')
		}
		return new Date(date);
	}

	/**
	 * 获取某年某月第一天是星期几
	 * @param year
	 * @param month
	 */
	static getWeekDay(year: number, month: number): number {
		const date = new Date(year, month - 1, 1);
		return date.getDay();
	}

	/**
	 * 格式化数字符串
	 * @param num
	 */
	static formatNumber(num: number): string {
		return num > 9 ? num + '' : "0" + num;
	}

	static isToday(date: DateType): boolean {
		let currentDate = new Date();
		let orderDate = DateHelper.getDateInstance(date);
		return currentDate.getFullYear() === orderDate.getFullYear() && orderDate.getMonth() === currentDate.getMonth() && orderDate.getDate() === currentDate.getDate();
	}

//格式化日期 YYYY-MM-DD
	static getFormatDate(date: DateType): string {
		const dateInstance = DateHelper.getDateInstance(date);
		return dateInstance.getFullYear() + "-" + DateHelper.formatNumber(dateInstance.getMonth() + 1) + "-" + DateHelper.formatNumber(dateInstance.getDate());
	}

//获取当前日期
	static getCurrentFormatDate(): string {
		return DateHelper.getFormatDate(new Date())
	}

	/**
	 * 获取当前日期所在月份的天数
	 * @param date
	 */
	static getDaysInMonth(date: DateType): MonthDayNumber {
		const dateInstance = DateHelper.getDateInstance(date);
		const month = dateInstance.getMonth() + 1;
		return DateHelper.getMonthDays(dateInstance.getFullYear(), month);
	}


	/**
	 * 获取星期名称
	 * @param date 日期
	 * @param isEnglish true 表示是英文
	 */
	static getWeekName(date: DateType, isEnglish: boolean = false): string {
		const dateInstance = DateHelper.getDateInstance(date);
		const weekDay = dateInstance.getDay()
		if (weekDay === 0) {
			return isEnglish ? 'Sunday' : '周日';
		} else if (weekDay === 1) {
			return isEnglish ? 'Monday' : "周一";
		} else if (weekDay === 2) {
			return isEnglish ? 'Tuesday' : '周二';
		} else if (weekDay === 3) {
			return isEnglish ? 'Wednesday' : '周三';
		} else if (weekDay === 4) {
			return isEnglish ? 'Thursday' : '周四';
		} else if (weekDay === 5) {
			return isEnglish ? 'Friday' : '周五';
		} else if (weekDay === 6) {
			return isEnglish ? 'Saturday' : '周六';
		}
		return '';
	}

	/**
	 * 是否是闰年
	 * @param year
	 */
	static isLeapYear(year: number): boolean {
		return year % 400 === 0 || (year % 4 === 0 && year % 100 !== 0);
	}

	/**
	 * 获取指定月份天数
	 * @param year
	 * @param month
	 * @returns {number}
	 */
	static getMonthDays(year: number, month: number): MonthDayNumber {
		if (month === 2) {
			return DateHelper.isLeapYear(year) ? 29 : 28;
		}
		const bigMonths = [1, 3, 5, 7, 8, 10, 12]
		if (bigMonths.includes(month)) {
			return 31;
		}
		return 30;
	}

	/**
	 *获取下个月信息
	 * @param year
	 * @param month
	 */
	static getNextMonthInfo(year: number, month: number) {
		const newMonth = month >= 12 ? 1 : month + 1;
		const newYear = month >= 12 ? year + 1 : year;
		return {year: newYear, month: newMonth}
	}

	/**
	 * 获取上个月信息
	 * @param year
	 * @param month
	 */
	static getLastMonthInfo(year: number, month: number) {
		//获取上个月实际月份
		const newMonth = month == 1 ? 12 : month - 1;
		//获取实际年份
		const newYear = month == 1 ? year - 1 : year;
		return {year: newYear, month: newMonth}
	}

	static diffTime(startTime: number, endTime: number, unitTime: number): number {
		const diffTimeNum = endTime - startTime
		return Math.floor((diffTimeNum / unitTime) * 100) / 100
	}

	/**
	 *相差年份数
	 * @param startTime
	 * @param endTime
	 */
	static diffYears(startTime: number, endTime: number) {
		let number = DateHelper.diffMonths(startTime, endTime);
		return Math.floor(number / 12 * 100) / 100;
	}

	/**
	 * 相差月分数
	 * @param startTime
	 * @param endTime
	 */
	static diffMonths(startTime: number, endTime: number) {
		let miniTime = Math.min(startTime, endTime);
		let maxTime = Math.max(startTime, endTime);
		const startDate = DateHelper.getDateInstance(miniTime);
		const endDate = DateHelper.getDateInstance(maxTime);
		let diffYear = endDate.getFullYear() - startDate.getFullYear();
		let startMonth = startDate.getMonth() + 1;
		let endMonth = endDate.getMonth() + 1;
		if (endMonth < startMonth) {
			diffYear = diffYear - 1;
			endMonth = endMonth + 12;
		}
		return Number.parseInt((diffYear * 12 + endMonth - startMonth).toFixed(0))
	}

	/**
	 * 相差天数
	 * @param startTime
	 * @param endTime
	 */
	static diffDays(startTime: number, endTime: number) {
		return DateHelper.diffTime(startTime, endTime, dateConstants.dayMillis)
	}

	/**
	 * 相差小时数
	 * @param startTime
	 * @param endTime
	 */
	static diffHours(startTime: number, endTime: number) {
		return DateHelper.diffTime(startTime, endTime, dateConstants.hourMillis)
	}

	/**
	 * 相差分数
	 * @param startTime
	 * @param endTime
	 */
	static diffMinutes(startTime: number, endTime: number) {
		return DateHelper.diffTime(startTime, endTime, dateConstants.minuteMillis)
	}

	/**
	 * 相差月秒数
	 * @param startTime
	 * @param endTime
	 */
	static diffSeconds(startTime: number, endTime: number) {
		return DateHelper.diffTime(startTime, endTime, dateConstants.secondMillis)
	}


	/**
	 * 格式化时间
	 * @param date
	 * @param formatStyle
	 */
	static formatDateString(date: DateType, formatStyle: string): string {
		let format = String(formatStyle);
		if (formatStyle) {
			let dateInstance = DateHelper.getDateInstance(date);
			formatDateList.forEach(formatItem => {
				for (let i = 0, len = formatItem.symbolList.length; i < len; i++) {
					if (format.indexOf(formatItem.symbolList[i]) >= 0) {
						format = format.replace(formatItem.symbolList[i], formatItem.getValue(dateInstance))
						break;
					}
				}
			})
		}
		return format;
	}

}
