
type WatchFun = (newVal: any, oldVal: any) => void

export class ObjectHelper {
  /**
   * 克隆函数
   * @param func
   */
  static cloneFunction(func: Function): any {
    // const bodyReg = /(?<={)(.|\n)+(?=})/m;
    const bodyReg = new RegExp("(?<={)(.|\\n)+(?=})","m");
    // const paramReg = /(?<=\().+(?=\)\s+{)/;
    const paramReg = new RegExp("(?<=\\().+(?=\\)\\s+{)");
    const funcString = func.toString();
    const param = paramReg.exec(funcString);
    const body = bodyReg.exec(funcString);
    if (param) {
      const paramArr = param[0].split(',');
      return body && body[0] ? new Function(...paramArr, body[0]) : new Function(...paramArr)
    } else {
      return body && body[0] ? new Function(body[0]) : new Function();
    }
  }

  /**
   * 深度克隆对象
   * @param obj
   * @param cache
   */
  static cloneDeep(obj: object | Array<any>, cache?: WeakMap<object, object>): any {
    if (obj === null || obj === undefined) {
      return obj
    }
    if (obj instanceof Function) {
      return ObjectHelper.cloneFunction(obj)
    }
    if (typeof obj !== 'object') {
      return obj
    } else {
      if (obj instanceof Date) {
        return new Date(obj)
      }
      if (obj instanceof RegExp) {
        return new RegExp(obj)
      }
      if (obj instanceof Symbol) {
        return Symbol(obj.description);
      }
      if (obj instanceof Map) {
        let newMap = new Map<any, any>();
        obj.forEach((value, key) => {
          newMap.set(key, ObjectHelper.cloneDeep(value))
        })
        return newMap;
      }
      if (obj instanceof Set) {
        let newSet = new Set<any>();
        obj.forEach(((value, value2) => {
          newSet.add(ObjectHelper.cloneDeep(value))
        }))
        return newSet;
      }
      if (obj instanceof Array) {
        let newArray = new Array<any>();
        for (let i = 0, len = obj.length; i < len; i++) {
          newArray.push(ObjectHelper.cloneDeep(obj[i]));
        }
        return newArray;
      }
      if (obj instanceof Object) {
        let weakMap = cache ? cache : new WeakMap<object, object>()
        if (weakMap.get(obj)) {
          // 防止循环引用，程序进入死循环
          return cache.get(obj)
        }
        let newObject = {};
        for (let key in obj) {
          newObject[key] = ObjectHelper.cloneDeep(obj[key], weakMap)
        }
        weakMap.set(obj, newObject); // 缓存拷贝的对象，用于处理循环引用的情况
        return newObject;
      }
    }
  }

  /**
   * 将某个对象追加到当前对象中(扩展对象，类似java中的继承)
   * @param targetObject
   * @param sourceObject
   */
  static extendObject(targetObject: object, sourceObject: object): object {
    if (!sourceObject || typeof (sourceObject) !== 'object') {//源对象为空或者不是对象时
      return targetObject;
    }
    if (targetObject && typeof (sourceObject) === 'object') {
      Object.keys(sourceObject).forEach(sourceKey => {
        if (typeof (targetObject[sourceKey]) === 'undefined') {//当前目标没有定义
          targetObject[sourceKey] = sourceObject[sourceKey];
        } else if (typeof (targetObject[sourceKey]) === 'object') {//当前目标不为空而且是对象时
          ObjectHelper.extendObject(targetObject[sourceKey], sourceObject[sourceKey])
        }
      })
    }
    return targetObject;
  }

  /**
   * 判断列表是是否包含特定的对象
   * @param list
   * @param object
   */
  static containObject(list: Array<any>, object: any) {
    if (!list || list.length < 1) {
      return false
    }
    for (let i = 0, len = list.length; i < len; i++) {
      if (ObjectHelper.equalDeep(list[i], object)) {
        return true;
      }
    }
    return false;
  }

  /**
   * 深入比较对象(判断对象是否相等)
   * @param object_1
   * @param object_2
   */
  static equalDeep(object_1: any, object_2: any): boolean {
    let type1 = typeof object_1
    let type2 = typeof object_2
    if (type1 !== type2) {
      return false
    }
    if (type1 === 'string' || type1 === 'number' || type1 === 'undefined' || type1 === 'boolean') {
      return object_1 === object_2
    }

    if (type1 === 'object') {
      if (object_1 instanceof Array) {
        if (object_1.length !== object_1.length) {
          return false
        }
        if (!object_1.every(item => ObjectHelper.containObject(object_2, item))) {
          return false
        }
        return object_2.every(item => ObjectHelper.containObject(object_1, item));
      } else {
        let keys_1 = Object.keys(object_1)
        let keys_2 = Object.keys(object_2)
        if (keys_1.length !== keys_2.length) {
          return false
        }
        let filterKeys = keys_2.filter(key => keys_1.includes(key))
        if (filterKeys.length !== keys_1.length) {
          return false
        }
        for (let i = 0, len = keys_1.length; i < len; i++) {
          let equal = ObjectHelper.equalDeep(object_1[keys_1[i]], object_2[keys_1[i]])
          if (!equal) {
            return false
          }
        }
        return true
      }
    }
    return false;
  }

  /**
   *
   * @param keysList 取值字段列表 ["a","b"]
   * @param valuesObject 目标对象｛a:"a",b:"11",c:"aaaa",d:"dd"}
   * @return  {} //如｛a:"a",b:"11"}
   */
  static getObjectByList(keysList: Array<string>, valuesObject: object): object {
    let newObject = {}
    if (keysList && keysList.length > 0 && valuesObject && typeof valuesObject === 'object') {
      for (let i = 0, len = keysList.length; i < len; i++) {
        let key = keysList[i];
        if (key) {
          newObject[key] = valuesObject[key]
        }
      }
    }
    return newObject
  }

  /**
   * 根据当前对象的key,从目标对象中取值
   * @param keysObject {a:""}
   * @param targetObject 目标对象｛a:"a",b:"11",c:"aaaa",d:"dd"}
   * @return  {} //如｛a:"a"}
   */
  static getObjectByObject(keysObject: object, targetObject: object): object {
    if (keysObject && targetObject && typeof keysObject === 'object' && typeof targetObject === 'object') {
      let keys = Object.keys(keysObject)
      return ObjectHelper.getObjectByList(keys, targetObject);
    }
    return {};
  }

  /**
   * 从对象中获取值 （支持 "a.b.c" 获取值）
   * @param obj
   * @param keyWords
   */
  static getValueFromObject(obj: object, keyWords): any {
    if (typeof obj === 'object' && keyWords && typeof keyWords === 'string') {
      if (obj[keyWords]) {
        return obj[keyWords];
      }
      let indexOf = keyWords.indexOf(".");
      if (indexOf < 0) {
        return obj[keyWords]
      } else {
        const keys = keyWords.split(".")
        if (obj[keys[0]] && obj[keys[0]] instanceof Object) {
          const nextKeyWorks = keys.slice(1).join('.')
          return ObjectHelper.getValueFromObject(obj[keys[0]], nextKeyWorks);
        }
      }
    }
    return "";
  }
	/**
	 * 设置监听器
	 */
	static addWatcher(data: object, watch: Record<string, (...args: any[]) => any>) {
		Object.keys(watch).forEach(v => {
			ObjectHelper.observe(data, v, watch[v]);
		})
	}

	/**
	 * 监听属性 并执行监听函数
	 */
	static observe(obj: object, key: string, watchFun: WatchFun): void {
		if (obj && typeof obj === 'object') {
			let val = obj[key]; //给该属性设默认值
			if (val) {
				Object.defineProperty(obj, key, {
					configurable: true,
					enumerable: true,
					set: function (value) {
						let oldVal = val;
						val = value;
						watchFun(value, oldVal);
					},
					get: function () {
						return val;
					}
				})
			} else {
				let indexOf = key.indexOf(".");
				if (indexOf > 0) {
					const objKey = key.substring(0, indexOf)
					let newKey = key.substring(indexOf + 1);
					ObjectHelper.observe(obj[objKey], newKey, watchFun)
				}
			}
		}
	}

}
