export class StringHelper {
	/**
	 * 替换所有
	 * @param content 当前文本
	 * @param pattern 旧字符串或者正则表达式
	 * @param newStr 要替换的内容
	 * @returns {*}  返回新的字符串
	 */
	static replaceAll(content: string, pattern: string, newStr: string) {
		const reg = new RegExp(pattern, "g"); //创建正则RegExp对象
		return content.replace(reg, newStr)
	}

	static isEmpty(str: string): boolean {
		return !str || str.trim().length < 1;
	}

	/**
	 * 是否是组织机构代码
	 * @param code
	 * @returns {*|boolean}
	 */
	static isOrganizationCode(code: string): boolean {
		// return /[0-9A-HJ-NPQRTUWXY]{8}-[0-9A-HJ-NPQRTUWXY]/g.test(code)
		return new RegExp("^[0-9A-HJ-NPQRTUWXY]{8}-[0-9A-HJ-NPQRTUWXY]$").test(code)
	}

	/**
	 * 18位的统一信用代码
	 * @param code
	 * @returns {*|boolean}
	 */
	static isUnifiedCreditCode(code: string): boolean {
		// return /^[0-9A-HJ-NPQRTUWXY]{2}\d{6}[0-9A-HJ-NPQRTUWXY]{10}$/.test(code)
		return new RegExp("^[0-9A-HJ-NPQRTUWXY]{2}\\d{6}[0-9A-HJ-NPQRTUWXY]{10}$").test(code)
	}

	/**
	 * 校验手机号（严格校验）
	 * //移动号段：134 135 136 137 138 139 147 148 150 151 152 157 158 159  165 172 178 182 183 184 187 188 195 198
	 * // 联通号段：130 131 132 145 146 155 156 166 167 171 175 176 185 186
	 * // 电信号段：133 149 153 173 174 177 180 181 189  191  199
	 * // 虚拟运营商:170
	 * 分组如下
	 * //130 131 132 133 134 135 136 137 138 139
	 * //145 146 147 148 149
	 * //150 151 152 153 155 156 157 158 159
	 * // 165 166 167
	 * //170  171  172 173 174  175 176 177 178
	 * //180 181 182 183 184 185 186 187 188 189
	 * // 191  195  198 199
	 * @param phone
	 * @returns {boolean}
	 */
	static isPhoneNumber(phone: number | string): boolean {
		const phoneStr = String(phone);
		//手机号正则
		// if (/^1[38][0-9]{9}$/.test(phoneStr)) {//校验//130 131 132 133 134 135 136 137 138 139  180 181 182 183 184 185 186 187 188 189号段
		if (new RegExp("^1[38][0-9]{9}$").test(phoneStr)) {//校验//130 131 132 133 134 135 136 137 138 139  180 181 182 183 184 185 186 187 188 189号段
			return true;
		}
		// if (/^14[56789][0-9]{8}$/.test(phoneStr)) {//校验14开头的号段
		if (new RegExp("^14[56789][0-9]{8}$").test(phoneStr)) {//校验14开头的号段
			return true
		}
		// if (/^15[012356789][0-9]{8}$/.test(phoneStr)) {//校验 1//150 151 152 153 155 156 157 158 159 开头的号段
		if (new RegExp("^15[012356789][0-9]{8}$").test(phoneStr)) {//校验 1//150 151 152 153 155 156 157 158 159 开头的号段
			return true;
		}
		// if (/^16[567][0-9]{8}$/.test(phoneStr)) {//校验 // 165 166 167 开头的号段
		if (new RegExp("^16[567][0-9]{8}$").test(phoneStr)) {//校验 // 165 166 167 开头的号段
			return true;
		}
		// if (/^17[012345678][0-9]{8}$/.test(phoneStr)) {//校验 //170  171  172 173 174  175 176 177 178 开头的号段
		if (new RegExp("^17[012345678][0-9]{8}$").test(phoneStr)) {//校验 //170  171  172 173 174  175 176 177 178 开头的号段
			return true;
		}
		// if (/^19[1589][0-9]{8}$/.test(phoneStr)) {//校验// 191  195  198 199 开头的号段
		if (new RegExp("^19[1589][0-9]{8}$").test(phoneStr)) {//校验// 191  195  198 199 开头的号段
			return true;
		}
		return false;
	}

	/**
	 * 判断是否是邮件地址
	 * @param email
	 * @returns {boolean}
	 */
	static isEmail(email: string): boolean {
		if (email) {
			// const regex = /^[a-zA-Z0-9]+([-_.][a-zA-Z0-9]+)*@[a-zA-Z0-9]+([-_.][a-zA-Z0-9]+)*\.[a-z]{2,}$/
			const regex = new RegExp("^[a-zA-Z0-9]+([-_.][a-zA-Z0-9]+)*@[a-zA-Z0-9]+([-_.][a-zA-Z0-9]+)*\\.[a-z]{2,}$")
			return regex.test(email)
		}
		return false
	}

	/**
	 * 获取文本（过滤副文本标签）
	 * @param richText
	 */
	static filterRichText(richText: string): string {
		if (!richText) {
			return ""
		}
		const regExp = new RegExp('<[^<>]+>', 'g');
		return richText.replace(regExp, "");
	}

	/**
	 * 从富文本中，查找img 标签，收集图片地址(只支持.png .jpg .jpeg 结尾)
	 * @param text 富文本
	 * @returns {[]}  返回图片列表
	 */
	static getImageListFromRichText(text: string): Array<string> {
		const list = new Array<string>();
		// let regular = /<img[^>]*>/g
		let regular = new RegExp("<img[^>]*>","g")
		let keysWords = text.match(regular)
		if (keysWords && keysWords.length > 0) {
			let srcReg =new RegExp("http\\S*((.jpg)|(.jpeg)|(.png))",'g')
			// let srcReg = /http\S*((.jpg)|(.jpeg)|(.png))/g
			keysWords.forEach(words => {
				let strings = words.match(srcReg);
				if (strings && strings.length > 0 && !list.includes(strings[0])) {
					list.push(strings[0])
				}
			})
		}
		return list;
	}

	/*RGB颜色转换为16进制*/
	static colorRbgToHex(rgbColor: string): string {
		let lowerCaseColor = rgbColor.toLowerCase();
		// let reg = /^#([0-9a-fA-f]{3}|[0-9a-fA-f]{6})$/;
		let reg = new RegExp("^#([0-9a-fA-f]{3}|[0-9a-fA-f]{6 })$");
		// if (/^(rgb|RGB)/.test(lowerCaseColor)) {
		if (new RegExp("^(rgb|RGB)").test(lowerCaseColor)) {
			let regExp = new RegExp("(?:\\(|\\)|rgb|RGB)*","g");
			// let aColor = lowerCaseColor.replace(/(?:\(|\)|rgb|RGB)*/g, "").split(",");
			let aColor = lowerCaseColor.replace(regExp, "").split(",");
			let strHex = "#";
			for (let i = 0; i < aColor.length; i++) {
				let hex = Number(aColor[i]).toString(16);
				if (hex === "0") {
					hex += hex;
				}
				strHex += hex;
			}
			if (strHex.length !== 7) {
				strHex = lowerCaseColor;
			}
			return strHex;
		} else if (reg.test(lowerCaseColor)) {
			// let aNum = lowerCaseColor.replace(/#/, "").split("");
			let aNum = lowerCaseColor.replace("#", "").split("");
			if (aNum.length === 6) {
				return lowerCaseColor;
			} else if (aNum.length === 3) {
				let numHex = "#";
				for (let i = 0; i < aNum.length; i += 1) {
					numHex += (aNum[i] + aNum[i]);
				}
				return numHex;
			}
		}
		return lowerCaseColor;
	}

	/**
	 *16进制颜色转为RGB格式
	 * @param hexColor  16进制颜色
	 * @returns {string}  RGB格式颜色
	 */
	static colorHexToRgb(hexColor: string): string {
		let sColor = hexColor.toLowerCase();
		// let reg = /^#([0-9a-fA-f]{3}|[0-9a-fA-f]{6})$/;
		let regExp = new RegExp("^#([0-9a-fA-f]{3}|[0-9a-fA-f]{6})$");
		if (sColor && regExp.test(sColor)) {
			if (sColor.length === 4) {
				let sColorNew = "#";
				for (let i = 1; i < 4; i += 1) {
					sColorNew += sColor.slice(i, i + 1).concat(sColor.slice(i, i + 1));
				}
				sColor = sColorNew;
			}
			//处理六位的颜色值
			let sColorChange = [];
			for (let i = 1; i < 7; i += 2) {
				sColorChange.push(parseInt("0x" + sColor.slice(i, i + 2)));
			}
			return "rgb(" + sColorChange.join(",") + ")";
		}
		return sColor;
	};

	/**
	 * 将整数部分逢三一断
	 * @param value
	 * @returns {string}
	 */
	static formatNumber(value: String | number): string {
		if (!value) {
			return '0'
		}
		let strings = String(value).split('.');
		// const leftNum = String(strings[0]).replace(/(\d)(?=(?:\d{3})+$)/g, '$1,');
		let regExp = new RegExp("(\\d)(?=(?:\\d{3})+$)");
		const leftNum = String(strings[0]).replace(regExp, '$1,');
		if (strings.length > 1) {
			return `${leftNum}.${strings[1]}`
		}
		return leftNum;
	}

	/**
	 * 通用脱敏
	 *
	 * 中文名字：只展示姓名第一个字
	 * 手机号：前三后四不脱敏，中间脱敏
	 * 身份证号：前6后4不脱敏，年月日脱敏
	 * 银行卡：前6位和后4位不脱敏，中间脱敏
	 * 其他 前6位和后4位不脱敏
	 * @param codeStr
	 * @return {string}
	 */
	static desensitize(codeStr) {
		// if (/[\u4e00-\u9fa5]/.test(codeStr)) {
		if (new RegExp("[\u4e00-\u9fa5]").test(codeStr)) {
			return codeStr[0] + "***"
		}
		const str = String(codeStr)
		let length = str.length;
		if (length === 11) {//脱敏手机
			return `${str.slice(0, 3)}****${str.slice(7, 11)}`
		}
		let regExp = new RegExp("^(.{6})(?:\\d+)(.{4})$");
		//身份证、银行卡
		// return str.replace(/^(.{6})(?:\d+)(.{4})$/, "\$1********\$2");
		return str.replace(regExp, "\$1********\$2");
	}

	/**
	 * 生成唯一的uuid;
	 */
	static generateUUID(): string {
		let d = new Date().getTime();
		return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c: string) {
			const r = (d + Math.random() * 16) % 16 | 0;
			d = Math.floor(d / 16);
			return (c == 'x' ? r : (r & 0x3 | 0x8)).toString(16);
		});
	}
}
