import {InterceptorsConfig} from "./InterceptorsConfig";
import {ResponseInterceptor} from './ResponseInterceptor'

const Method = {
	GET: 'get',
	POST: 'post',
	DELETE: 'delete',
	HEAD: 'head',
	CONNECT: 'CONNECT',
	OPTIONS: 'options',
	PUT: 'put',
	PATCH: 'patch',
	TRACE: 'TRACE',
	LINK: 'LINK',
	UNLINK: 'unlink',
}
export default class HttpClient {
	private readonly baseUrl: string;
	private readonly platform: object;
	private readonly requestInterceptorsConfig: InterceptorsConfig;
	private readonly taskMap: Map<string, any>;
	private readonly isMiniApp: boolean;

	constructor(platform: object, isMiniApp: boolean, baseUrl: string, requestInterceptorsConfig: InterceptorsConfig) {
		this.platform = platform;
		this.baseUrl = baseUrl;
		this.requestInterceptorsConfig = requestInterceptorsConfig;
		this.taskMap = new Map<string, any>();
		this.isMiniApp = isMiniApp;
	}

	post<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.POST, params, headers, cancelToken)
	}

	get<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		const requestUrl = HttpClient.getTargetUrl(path, params);//get请求得手动拼接参数
		return this.request(requestUrl, Method.GET, {}, headers, cancelToken)
	}

	put<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.PUT, params, headers, cancelToken)
	}

	delete<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.DELETE, params, headers, cancelToken)
	}

	head<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.HEAD, params, headers, cancelToken)
	}

	options<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.OPTIONS, params, headers, cancelToken)
	}

	connect<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.CONNECT, params, headers, cancelToken)
	}

	trace<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.TRACE, params, headers, cancelToken)
	}

	patch<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.PATCH, params, headers, cancelToken)
	}

	link<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.LINK, params, headers, cancelToken)
	}

	unlink<T>(path: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		return this.request(path, Method.UNLINK, params, headers, cancelToken)
	}

	private static getTargetUrl(baseUrl: string, params: object): string {
		if (params && Object.keys(params).length > 0) {
			let hasStartFlag = baseUrl.indexOf('?') !== -1;
			let requestUtl = baseUrl
			for (let key in params) {
				if (params[key] !== undefined) {
					if (requestUtl.endsWith("?")) {
						requestUtl = `${requestUtl}${key}=${params[key]}`
					} else {
						requestUtl = `${requestUtl}${!hasStartFlag ? '?' : '&'}${key}=${params[key]}`
						if (!hasStartFlag) {
							hasStartFlag = true
						}
					}
				}
			}
			return requestUtl
		}
		return baseUrl
	}

	private request<T>(url: string, method: string, params?: any, headers?: any, cancelToken?: string): Promise<T> {
		if (this.requestInterceptorsConfig && typeof this.requestInterceptorsConfig.getCacheData === "function") {
			let cacheData = this.requestInterceptorsConfig.getCacheData(this.baseUrl, url, params);
			if (cacheData !== undefined) {
				return Promise.resolve(cacheData)
			}
		}
		const defaultHeaders = {
			'Content-Type': 'application/json',
			...headers
		}
		//通过请求拦截器,获取统一的自定义请求头
		const headerConfig = this.requestInterceptorsConfig && this.requestInterceptorsConfig.onRequestInterceptor ? this.requestInterceptorsConfig.onRequestInterceptor(defaultHeaders) : defaultHeaders;
		const requestUrl = this.getRequestUrl(url);
		const defaultOptions = {
			baseURL: '',
			url: requestUrl,
			method: method,
			data: params,
			header: {
				...headerConfig
			}
		}
		// @ts-ignore
		if (this.isMiniApp && this.platform && this.platform.request && typeof this.platform.request === 'function') {
			return this.miniAppRequest(defaultOptions, cancelToken, url, params);
		} else if (this.platform && typeof this.platform === 'function') {
			return this.commonRequest(defaultOptions, cancelToken, url, params)
		}
	}

	private getRequestUrl(url: string): string {
		if (url.startsWith('https://') || url.startsWith('http://')) {
			return url;
		}
		if (this.baseUrl) {
			let endsWith = this.baseUrl.endsWith("/");
			let startsWith = url.startsWith('/');
			if (endsWith && startsWith) {
				return this.baseUrl.substring(0, this.baseUrl.length - 1) + url;
			} else if (!endsWith && !startsWith) {
				return this.baseUrl + '/' + url;
			}
			return this.baseUrl + url
		}
		return url;
	}

	private miniAppRequest<T>(options: object, cancelToken: string | undefined, url: string, params?: any): Promise<T> {
		return new Promise((resolve, reject) => {
			const requestOptions = {
				...options,
				fail: (res: any) => {
					if (cancelToken) {
						this.taskMap.delete(cancelToken);
					}
					reject(res);
				},
				success: (res: any) => {
					if (cancelToken) {
						this.taskMap.delete(cancelToken);
					}
					this.successHandler(resolve, reject, res, url, params);
				}
			}
			// @ts-ignore
			let requestTask = this.platform.request(requestOptions);
			if (cancelToken && requestTask) {
				this.taskMap.set(cancelToken, requestTask)
			}
		})
	}

	private successHandler<T>(resolve: (value: T | PromiseLike<T>) => void, reject: (reason?: any) => void, res: any, url: string, params: any) {
		let responseData = this.getResponseData(res, url, params);
		if (responseData.success) {
			resolve(responseData.data);
		} else {
			reject(responseData.data);
		}
	}

	private getResponseData(res: any, url: string, params?: any): ResponseInterceptor {
		let responseData = {success: true, data: res};
		if (this.requestInterceptorsConfig && typeof this.requestInterceptorsConfig.onResponseInterceptor === 'function') {
			responseData = this.requestInterceptorsConfig.onResponseInterceptor(res);
		}
		if (responseData.success && this.requestInterceptorsConfig && typeof this.requestInterceptorsConfig.updateResponseCache === 'function') {
			this.requestInterceptorsConfig.updateResponseCache(res, this.baseUrl, url, params);//缓存数据
		}
		return responseData;
	}

	private commonRequest<T>(options: object, cancelToken: string | undefined, url: string, params?: any): Promise<T> {
		return new Promise((resolve, reject) => {
			// @ts-ignore
			this.platform(options)
					.then((res: any) => {
						this.successHandler(resolve, reject, res, url, params);
					})
					.catch((reason: any) => {
						reject(reason);
					})
		})
	}
}
